/**
 * S9H Store Services JavaScript
 * 
 * Handles the client-side functionality for the S9H Store Services & Clients page,
 * including form submissions and CRUD operations for services and clients.
 * 
 * @package Cockpit
 * @subpackage S9HStore
 * @author Ian O'Neill
 * @version See version.json
 */

/**
 * Get the CSRF token from the page
 * 
 * @returns {string} The CSRF token value
 */
function getCsrfToken() {
    return $('input[name="csrf_token"]').val();
}

/**
 * Populate the service form with existing service data for editing
 * 
 * @param {string} name - The service name
 * @param {string} url - The service URL
 */
function editService(name, url) {
    $("#name").val(name);
    $("#url").val(url);
    $("#name").focus();

    $('#post_service button[name="post_service"]').text('Update Service');

    // Scroll to the form smoothly
    $('html, body').animate({
        scrollTop: $("#post_service").offset().top - 100
    }, 500);
}

/**
 * Delete a service after confirmation
 * 
 * @param {number} id - ID of the service to delete
 */
function deleteService(id) {
    if (!id || isNaN(id)) {
        console.error('Invalid service ID:', id);
        return;
    }

    if (confirm("Are you sure you want to delete this service? This action cannot be undone.")) {
        $.ajax({
            type: "POST",
            url: "./includes/post_s9hstore_services.php",
            data: {
                delete_service: 1,
                id: id,
                csrf_token: getCsrfToken()
            },
            success: function () {
                location.reload();
            },
            error: function (xhr, status, error) {
                console.error('Error deleting service:', status, error);
            }
        });
    }
}

/**
 * Populate the client form with existing client data for editing
 * 
 * @param {string} username - The client username
 * @param {string} password - The client password
 * @param {string} expiry - The client expiration date
 */
function editClient(username, password, expiry) {
    $("#username").val(username);
    $("#password").val(password);
    $("#expiration").val(expiry);
    $("#username").focus();

    $('#post_client button[name="post_client"]').text('Update Client');

    // Scroll to the form smoothly
    $('html, body').animate({
        scrollTop: $("#post_client").offset().top - 100
    }, 500);
}

/**
 * Delete a client after confirmation
 * 
 * @param {number} id - ID of the client to delete
 */
function deleteClient(id) {
    if (!id || isNaN(id)) {
        console.error('Invalid client ID:', id);
        return;
    }

    if (confirm("Are you sure you want to delete this client? This action cannot be undone.")) {
        $.ajax({
            type: "POST",
            url: "./includes/post_s9hstore_services.php",
            data: {
                delete_client: 1,
                id: id,
                csrf_token: getCsrfToken()
            },
            success: function () {
                location.reload();
            },
            error: function (xhr, status, error) {
                console.error('Error deleting client:', status, error);
            }
        });
    }
}

/**
 * Ban a client after confirmation
 * 
 * @param {number} id - ID of the client to ban
 */
function banClient(id) {
    if (!id || isNaN(id)) {
        console.error('Invalid client ID:', id);
        return;
    }

    if (confirm("Are you sure you want to ban this client? They will not be able to access the service.")) {
        $.ajax({
            type: "POST",
            url: "./includes/post_s9hstore_services.php",
            data: {
                ban_client: 1,
                id: id,
                csrf_token: getCsrfToken()
            },
            success: function () {
                location.reload();
            },
            error: function (xhr, status, error) {
                console.error('Error banning client:', status, error);
            }
        });
    }
}

/**
 * Unban a client after confirmation
 * 
 * @param {number} id - ID of the client to unban
 */
function unbanClient(id) {
    if (!id || isNaN(id)) {
        console.error('Invalid client ID:', id);
        return;
    }

    if (confirm("Are you sure you want to unban this client? They will regain access to the service.")) {
        $.ajax({
            type: "POST",
            url: "./includes/post_s9hstore_services.php",
            data: {
                unban_client: 1,
                id: id,
                csrf_token: getCsrfToken()
            },
            success: function () {
                location.reload();
            },
            error: function (xhr, status, error) {
                console.error('Error unbanning client:', status, error);
            }
        });
    }
}

/**
 * Generate a random alphanumeric string
 * @param {number} length - Length of the string to generate
 * @returns {string} Random string
 */
function generateRandomString(length = 8) {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    let result = '';
    for (let i = 0; i < length; i++) {
        result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return result;
}

/**
 * Copy text to clipboard
 * @param {string} text - Text to copy
 */
function copyToClipboard(text) {
    if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(text).then(() => {
            showToast('Success', 'Copied to clipboard!', 'success');
        }).catch(err => {
            console.error('Failed to copy: ', err);
            fallbackCopyToClipboard(text);
        });
    } else {
        fallbackCopyToClipboard(text);
    }
}

/**
 * Fallback copy method for older browsers
 * @param {string} text - Text to copy
 */
function fallbackCopyToClipboard(text) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    document.body.appendChild(textArea);
    textArea.select();
    try {
        document.execCommand('copy');
        showToast('Success', 'Copied to clipboard!', 'success');
    } catch (err) {
        console.error('Fallback copy failed: ', err);
        showToast('Error', 'Failed to copy', 'danger');
    }
    document.body.removeChild(textArea);
}

/**
 * Show toast notification
 * @param {string} title - Toast title
 * @param {string} message - Toast message
 * @param {string} type - Toast type (success, danger, warning, info)
 */
function showToast(title, message, type = 'info') {
    const container = document.querySelector('.toasts-container');
    if (!container) return;

    const iconMap = {
        success: 'fa-check-circle text-theme',
        danger: 'fa-exclamation-circle text-danger',
        warning: 'fa-exclamation-triangle text-warning',
        info: 'fa-info-circle text-info'
    };

    const toastHtml = `
        <div class="toast fade show mb-3" role="alert" aria-live="assertive" aria-atomic="true" data-bs-autohide="true" data-bs-delay="3000">
            <div class="toast-header">
                <i class="fas ${iconMap[type] || iconMap.info} me-2"></i>
                <strong class="me-auto">${title}</strong>
                <button type="button" class="btn-close" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
            <div class="toast-body">
                ${message}
            </div>
        </div>
    `;

    container.insertAdjacentHTML('beforeend', toastHtml);

    // Auto-remove toast after delay
    setTimeout(() => {
        const toasts = container.querySelectorAll('.toast');
        if (toasts.length > 0) {
            toasts[0].remove();
        }
    }, 3500);
}

/**
 * Copy client credentials in formatted text
 * @param {string} username - Client username
 * @param {string} password - Client password
 */
function copyClientCredentials(username, password) {
    const text = `Username: ${username}\nPassword: ${password}`;
    copyToClipboard(text);
}

// Initialize on document ready
$(function () {
    // Initialize toasts with auto-hide
    $('.toast').each(function() {
        var toast = new bootstrap.Toast(this);
        toast.show();
    });
    
    /**
     * Initialize datepicker with proper configuration
     */
    $('#expiration').datepicker({
        autoclose: true,
        format: 'yyyy-mm-dd',
        todayHighlight: true,
        orientation: 'bottom auto',
        clearBtn: true
    });

    /**
     * Service form submission handler
     */
    $("#post_service").on("submit", function (e) {
        e.preventDefault();

        const formData = new FormData(this);
        formData.append('post_service', '1');
        formData.append('csrf_token', getCsrfToken());

        // Disable submit button to prevent double submission
        const submitBtn = $(this).find('button[type="submit"]');
        const originalText = submitBtn.text();
        submitBtn.prop('disabled', true).text('Saving...');

        $.ajax({
            type: "POST",
            url: "./includes/post_s9hstore_services.php",
            data: formData,
            processData: false,
            contentType: false,
            success: function () {
                location.reload();
            },
            error: function (xhr, status, error) {
                console.error('Error submitting service:', status, error);
                submitBtn.prop('disabled', false).text(originalText);
            }
        });
    });

    /**
     * Client form submission handler
     */
    $("#post_client").on("submit", function (e) {
        e.preventDefault();

        const formData = new FormData(this);
        formData.append('post_client', '1');
        formData.append('csrf_token', getCsrfToken());

        // Disable submit button to prevent double submission
        const submitBtn = $(this).find('button[type="submit"]');
        const originalText = submitBtn.text();
        submitBtn.prop('disabled', true).text('Saving...');

        $.ajax({
            type: "POST",
            url: "./includes/post_s9hstore_services.php",
            data: formData,
            processData: false,
            contentType: false,
            success: function () {
                location.reload();
            },
            error: function (xhr, status, error) {
                console.error('Error submitting client:', status, error);
                submitBtn.prop('disabled', false).text(originalText);
            }
        });
    });

    /**
     * Generate username button
     */
    $('#generateUsernameBtn').on('click', function(e) {
        e.preventDefault();
        const username = generateRandomString(8);
        $('#username').val(username);
    });

    /**
     * Generate password button
     */
    $('#generatePasswordBtn').on('click', function(e) {
        e.preventDefault();
        const password = generateRandomString(8);
        $('#password').val(password);
    });

    /**
     * Copy credentials button (from form)
     */
    $('#copyCredentialsBtn').on('click', function(e) {
        e.preventDefault();
        const username = $('#username').val();
        const password = $('#password').val();
        
        if (!username || !password) {
            showToast('Warning', 'Please enter username and password first', 'warning');
            return;
        }
        
        copyClientCredentials(username, password);
    });
});